/*!************************************************************************
 * FILE :         clAudioSMEngine.cpp
 *
 * SW-COMPONENT:  Audio statemachine
 *
 * DESCRIPTION:
 *                STARTUP:
 *                StraightForward Case is:
 *                - Check in persistent DP for LSM and PSM
 *                  LSM: Last Selected Mode, i.e. the Entertainment Source that was
 *                  played before switching off the system
 *                  PSM: Previous Selected Mode, i.e. the Entertainment Source that was
 *                  played before the LSM
 *                - Check if LSM still available
 *                  If LSM available: start it
 *                  If LSM not available: Check if PSM still available
 *                     If PSM available: start it
 *                     If PSM not available: Start Radio.FM1
 *                Startup Details:
 *                - Before any communication to FC-Audio is initiated, wait for
 *                  AudioService to become available.
 *                - Before the availability of a media source can be checked, wait for
 *                  MediaService to become available and wait, until information about
 *                  removable mediasources is available. (was SD removed from slot while off?)
 *                - Prevent, that the XML framework begins to place API-Calls like:
 *                  Sound.GetCurrentAudioComponent()
 *                  AVDC.GetCurrentSource()
 *                  AVDC.Audio.GetAudioSourceState(...)
 *                  before the mentioned Services are available.
 *                - Before starting LSM, ensure, that media-removed info is evaluated on stack
 *                  Don't try to start e.g. LSM:SD and then immediately stop it, because is was removed
 *                  during off-state.
 *
 *                  DATAPOOL:
 *                  The following DP entries are relevant for Audio Source Switching
 *                  and the corresponding Screen-Changes / visual animations:
 *                  DPAUDIO__PER_ENTSRC:
 *                     Stores the current Audio Component (Radio or Media)
 *                     Used to determine/switch the menu net Radio vs. Media
 *                     Not persistent.
 *                  DPAUDIO__ENT_SRC:
 *                     Stores the current Radio-Band. (AM, FM, DAB)
 *                     Used to determine/switch the screen in Radio-Menu
 *                     In addition this value is required to maintain the history
 *                     Persistent.
 *                  DPAVDC__CURSRC:
 *                     Stores the current Media-Source (SD, CD-DA, CDC, etc)
 *                     Used to determine/switch the screen in Media-Menu
 *                     In addition this value is required to maintain the history
 *                     Persistent.
 *                  DPAVDC_PREV_AUDIO_SRC and DPAVDC_PCUR_AUDIO_SRC:
 *                     Stores the current and previous (Entertainment-)Audio-Source
 *                     Not used for Screen Changes. No GUI-Data Events.
 *                     Used for the history-stack for entertainment sources in the Audio-handler
 *                     Persistent.
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steilmann
 *
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Version
 *  23.10.07       CM-DI/PJ-VW34 Steinle
 *                 moved Creation of AudioSourceController from Contructor
 *                 to bConfigure.
 * 06.03.2008      CM-DI/PJ-VW34 Steinle
 *                 AudioEngine requests are allowed now also when AudioEngine is not started
 *                 This is posibble due to the Srv-Unavailable-handling in
 *                 the audio-src-implementations
 * 07.03.2008      CM-DI/PJ-VW34 Steinle
 *                 fix for mediaSourceRemoved-Event. If PSM=radio-am
 *                 goto am (not fm). Solved with isEntertainmentsourceAvailable()
 * 08.03.2008      CM-DI/PJ-VW34 Steinle
 *                 introduced persistence for mute-state and separate mute-pin source
 * 17.07.2013      CM-AI/CB-3 R.Volkmer (mailto:R.Volkmer@ITB-Solutions)
 *             initial extraction from HMI and integrate to Genivi AudioManager CtrlPlugIn
 *************************************************************************/

#include "AudioStack/clAudioSMEngine.h"
#include "AudioStack/AudioSources.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "AudioStack/AudioSources/clAudioSourceFactory.h"

#define DP_DATAPOOL_ID =0xFFF1
#define DP_S_IMPORT_INTERFACE_FI
//#include "dp_if.h"
#include "dp_fff1_if.h"


// Unorganised traces
#ifndef USE_DLT_TRACE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_COMP_AUDIOSTACK
#include "trcGenProj/Header/clAudioSMEngine.cpp.trc.h"
#endif
#endif

#define CFC_FI_S_IMPORT_INTERFACE_FI_TYPES
#include "cfc_fi_if.h"


#include "AudioStack/clTogglingSequences.h"
#include "AudioStack/clGeniviAudioCtrlAdapter.h"

namespace AudioStack
{

tU8                        clAudioSMEngine::m_u8ErrorCode                     = AUDIOSM_OK;
tU8                        clAudioSMEngine::m_u8AudioRouteMgr_SrvState        = 0;//AMT_C_U8_SVCSTATE_NOT_AVAILABLE;
tU8                        clAudioSMEngine::m_u8Runlevel                      = 0;
SourceID                   clAudioSMEngine::m_enSource_Entertainment_Previous = SourceID(AudioSources::NONE,0);
SourceID                   clAudioSMEngine::m_enSource_Entertainment_Active   = SourceID(AudioSources::NONE,0);
SourceID                   clAudioSMEngine::m_enSource_Entertainment_LSM      = SourceID(AudioSources::NONE,0);
time_t                     clAudioSMEngine::m_stInitTime                      = time(NULL);
/*!************************************************************************
 * METHOD:        constructor
 *
 * CLASS:         clAudioSMEngine
 *
 * DESCRIPTION:      default constructor.
 *
 * PARAMETER:
 *             none
 *
 * RETURNVALUE:
 *             none
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Revision
 *************************************************************************/
clAudioSMEngine::clAudioSMEngine()
{


}



/*!***********************************************************************
 * METHOD:        destructor
 *
 * CLASS:         clAudioSMEngine
 *
 * DESCRIPTION:      destructor.
 *
 * PARAMETER:
 *             none
 * RETURNVALUE:
 *             none
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Revision
 ************************************************************************/
clAudioSMEngine::~clAudioSMEngine()
{}



clAudioSMEngine::enErrorCode clAudioSMEngine::Source_On(SourceID srcID, tU32 u32UserData)
{
   if(((m_u8Runlevel < 2))) //
   {
      ETG_TRACE_USR1((" Source_On:  NOT allowed due to runlevel %d", m_u8Runlevel));
      return AUDIOSM_ERROR;
   }
   if(! clAudioSourceFactory::isValidSourceClass(srcID.enSourceClass))
   {
      ETG_TRACE_ERR(("Source_Off: SourceClass %d not available by AudioSourceFactory",
            ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))));
      return AUDIOSM_ERROR;
   }
   return AudioSourceOn(srcID, FALSE, u32UserData);
}

clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Off(SourceID source, tU32 u32UserData)
{
   if(((m_u8Runlevel < 2))) //
   {
      ETG_TRACE_USR1((" Source_Off:  NOT allowed due to runlevel %d", m_u8Runlevel));
      return AUDIOSM_ERROR;
   }
   if(! clAudioSourceFactory::isValidSourceClass(source.enSourceClass))
   {
      ETG_TRACE_ERR(("Source_Off: SourceClass %d not available by AudioSourceFactory",
            ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(source.enSourceClass))));
      return AUDIOSM_ERROR;
   }
   vAudioSourceOff(source, u32UserData);
   return AUDIOSM_OK;
}



clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Removed(SourceID source)
{
   // ignore request, as long as runlevel for media not reached
   if(m_u8Runlevel >= 3)
   {
      if(! clAudioSourceFactory::isValidSourceClass(source.enSourceClass))
      {
         ETG_TRACE_ERR(("Source_Removed: SourceClass %d not available by AudioSourceFactory",
               ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(source.enSourceClass))));
         return AUDIOSM_ERROR;
      }
      return SourceRemoved(source);
   }
   else
   {
      ETG_TRACE_USR1((" Source_Removed:  Ignored because of runlevel %d", m_u8Runlevel));
   }
   return AUDIOSM_ERROR;
}

clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Insert(SourceID source)
{
   // ignore request, as long as runlevel for media not reached
   if(m_u8Runlevel >= 3)
   {
      if(! clAudioSourceFactory::isValidSourceClass(source.enSourceClass))
      {
         ETG_TRACE_ERR(("Source_Insert: SourceClass %d not available by AudioSourceFactory",
               ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(source.enSourceClass))));
         return AUDIOSM_ERROR;
      }
      return SourceInserted(source);
   }
   else
   {
      ETG_TRACE_USR1((" Source_Insert:  Ignored because of runlevel %d", m_u8Runlevel));
   }
   return AUDIOSM_ERROR;
}


tVoid clAudioSMEngine::ResetStack()
{
   clAudioSourceController::getInstance().vResetStack();
}



tVoid clAudioSMEngine::TraceStack()
{
   clAudioSourceController::getInstance().vTraceStack();
}



/*************************************************************************
* METHOD:          vMediaSourceInserted
*
* DESCRIPTION:     A media source has been inserted
*                  So it should be started (regarding priority rules)
*                  This event is currently handled the same way like a
*                  SetSource() request from user.
* PARAMETER:       -u8MediaSource   The inserted media source
*
* RETURNVALUE:     -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::SourceInserted(SourceID srcID)
{
   ETG_TRACE_USR4(("SourceInserted: SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
         , srcID.u16SubSource
         , srcID.u16SubSource));
   return AudioSourceOn(srcID);
}

/*************************************************************************
* METHOD:          vMediaSourceRemoved
*
* DESCRIPTION:     If a MediaSource is removed/ejected by the user, we have
*                  to stop it if it is currently playing.
*                  Then we start another AudioSource, regarding the rules:
*                  If the previously played Source was Radio: start Radio
*                  If the previously played Source was Media:
*                     If the Media is still available: start that media
*                     If the Media is not available: start Radio
*
* PARAMETER:       tU32 u32MediaSourceToRemove
*                       the media source that was ejected/removed by the user
* RETURNVALUE:     -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::SourceRemoved(SourceID srcID)
{

   clAudioSource* pSrc = clAudioSourceFactory::getAudioSource(srcID);
   //Set source to unavailable
   if(pSrc==NULL)
   {
      ETG_TRACE_ERR(("ERROR SourceRemoved: no Instance for SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
         , srcID.u16SubSource
         , srcID.u16SubSource));
      return clAudioSMEngine::AUDIOSM_ERROR;
   }
   if(pSrc->enIsSourceAvailable() != clAudioSource::not_available)
   {
      pSrc->vSourceAvailablilityChange(clAudioSource::not_available,clAudioSource::nomedia);
   }

   ETG_TRACE_USR4(("SourceRemoved: SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
         , srcID.u16SubSource
         , srcID.u16SubSource));
   SourceID restoreSrcID = m_enSource_Entertainment_Previous;
   ETG_TRACE_USR4(("SourceRemoved: previous Entertainment SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(restoreSrcID.enSourceClass))
            , restoreSrcID.u16SubSource
            , restoreSrcID.u16SubSource));

   if(restoreSrcID.enSourceClass == AudioSources::NONE)
   {
      ETG_TRACE_USR1(("SourceRemoved: No previous Source, restoring FM"));
      restoreSrcID = SourceID(AudioSources::TUNER_FM,0);
   }

   if(!bIsSourceAvailable(restoreSrcID))
   {
      AudioToggleGroups::enToggleGroup toggleGroup =
            clTogglingSequences::FindToggleGroup(static_cast<AudioSources::enAudioSources>(srcID.enSourceClass));
      AudioSources::enAudioSources srcClass = clTogglingSequences::GetNextSourceClass(srcID.enSourceClass, toggleGroup);
      clAudioSource* pAudSrc = clAudioSourceFactory::getAvailableAudioSource(srcClass);
      if(pAudSrc != NULL)
      {
         restoreSrcID = pAudSrc->sGetId();
      }else{
         ETG_TRACE_USR1(("SourceRemoved: Restoring default source"));
         restoreSrcID = SourceID(AudioSources::TUNER_FM,0);
      }
   }
   clAudioSourceController::getInstance().vHandleSrcActError(srcID);
   clAudioSourceFactory::removeAudioSources(srcID);
   ETG_TRACE_USR4(("SourceRemoved: activating SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
            , srcID.u16SubSource
            , srcID.u16SubSource));
   AudioSourceOn(restoreSrcID, TRUE);

   return AUDIOSM_OK;
}

/*************************************************************************
* METHOD:         vHandleSrcActError
*
* DESCRIPTION:    Remove defect Source from AudioStack.
************************************************************************/
tVoid clAudioSMEngine::vHandleSrcActError(SourceID errSrc)
{

   SourceID prevAudioSrc = m_enSource_Entertainment_Previous;
   if(!bIsSourceAvailable(prevAudioSrc))
   {
      prevAudioSrc = SourceID(AudioSources::TUNER_FM,0);
   }
   ETG_TRACE_ERR(("vHandleSrcActError: Error at source: %d, fallback source: %d"
         ,ETG_CENUM(AudioSources::enAudioSources, errSrc)
         ,ETG_CENUM(AudioSources::enAudioSources, prevAudioSrc)));
   clAudioSourceController::getInstance().vHandleSrcActError(errSrc);
}

/*************************************************************************
* METHOD:         vAudioSourceOn
*
* DESCRIPTION:    Called on each request of a source change.
*                 This method handles requests for Entertainment-sources
*                 as well as requests for Foreground- and Mix-sources
*
*                 FOREGROUND SOURCES:
*                 - Just forward the request to the AudioSourceController
*                 - No update of Datapool entries
*                 - No update play history
*
*                 MIX SOURCES:
*                 - Just forward the request to the AudioSourceController
*                 - No update of Datapool entries
*                 - No update play history
*
*                 ENTERTANMENT SOURCES:
*                 - Immediately and synchronously update the Datapool in order
*                 to initiate the screen change and to start the visual
*                 animation. This operation has to be performed, independent
*                 of the operation on the AudioStack. I.e. we even update the
*                 Datapool.CurrentSource although we still do not know, if the
*                 new requested source is allowed to play.
*                 This is essential for entertainment sources as it might be
*                 required to change screens and menu-nets for entertainment-
*                 source while other (Foreground-)audiosources are active.
*                 In order to store the playhistory for LSM and PSM, the
*                 Datapool entries:
*                    DPAVDC__CUR_AUDIO_SRC
*                    DPAVDC__PREV_AUDIO_SRC
*                 are stored. Thes entries only contain entertainment sources!
*
* PARAMETER:       -
*
* RETURNVALUE:     -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::AudioSourceOn(SourceID srcID, tBool bRemoveCurrent, tU32 u32UserData)
{
   clAudioSource* poAudioSource = NULL;
   clStackRules::group_t enGroup;
   clStackRules::type_t enType;

   // u8AudioSource = static_cast<AudioSources::enAudioSources>(clAudioSourceFactory::mapVirtualToLogicalSource(static_cast<AudioSources::enAudioSources>(u8AudioSource)));
   if(! clAudioSourceFactory::isValidSourceClass(srcID.enSourceClass))
   {
      ETG_TRACE_ERR(("clAudioSMEngine::vAudioSourceOn,SourceClass %d not available by AudioSourceFactory",
            ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))));
      return AUDIOSM_ERROR;
   }

   ETG_TRACE_USR4(("vAudioSourceOn: SourceClass: %d subID %#x(%d), UsrData %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
         , (tU16)srcID.u16SubSource
         , (tU16)srcID.u16SubSource
         , (tU32)u32UserData
         , (tU32)u32UserData));

   poAudioSource = clAudioSourceFactory::getAudioSource(srcID);
   if (poAudioSource == NULL)
   {
      ETG_TRACE_USR1(("SourceClass: %d subID %#x(%d) not available by AudioSourceFactory"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
            , (tU16)srcID.u16SubSource
            , (tU16)srcID.u16SubSource));
      return AUDIOSM_ERROR;
   }
   enGroup = poAudioSource->getGroup();
   enType  = poAudioSource->getType();

   if (clAudioSourceController::getInstance().enCheckSource(poAudioSource) != clStackRules::allowed)
   {
      ETG_TRACE_USR1(("SourceClass: %d not allowed!",
            ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(poAudioSource->sGetId().enSourceClass))));
      return AUDIOSM_ERROR;
   }

   //For available and unknown we will put the source on stack
   if(poAudioSource->enIsSourceAvailable() == clAudioSource::not_available)
   {
      ETG_TRACE_USR1(("ERROR: SourceClass: %d subID %#x(%d): NOT AVAILABLE"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
         , (tU16)srcID.u16SubSource
         , (tU16)srcID.u16SubSource));
         return AUDIOSM_ERROR;
   }

   // ******
   // STEP 1
   // ******
   // Update the persistent history-stack of (persistent) Entertainment-Sources
   if (enType == clStackRules::typeBg)
   {
      if (srcID != m_enSource_Entertainment_Active)
      {
         if (bRemoveCurrent)
         {
            // if current bg-source should be removed, don't keep it in history-stack
            m_enSource_Entertainment_Previous = SourceID(AudioSources::NONE,0);
         }
         else
         {
            // the current bg- source will be saved in the history-stack as the previous source
            m_enSource_Entertainment_Previous = m_enSource_Entertainment_Active;
         }
         m_enSource_Entertainment_Active = srcID;

         ETG_TRACE_USR1(("Entertainment Source update: ACTIVE SourceClass: %d subID %#x(%d), PREVIOUS SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(m_enSource_Entertainment_Active.enSourceClass))
         , (tU16)m_enSource_Entertainment_Active.u16SubSource
         , (tU16)m_enSource_Entertainment_Active.u16SubSource
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(m_enSource_Entertainment_Previous.enSourceClass))
         , (tU16)m_enSource_Entertainment_Previous.u16SubSource
         , (tU16)m_enSource_Entertainment_Previous.u16SubSource));
      }
   }
   // ******
   // STEP 2
   // ******

   //Store persistently
   if(enType == clStackRules::typeBg)
   {
      // Update the DP values, which will trigger the screen-changes and animations
      ETG_TRACE_USR3(("vAudioSourceOn(): Step 2: Store Last Source SourceClass: %d subID %#x(%d) enGroup: %d"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(m_enSource_Entertainment_Active.enSourceClass))
            , (tU16)m_enSource_Entertainment_Active.u16SubSource
            , (tU16)m_enSource_Entertainment_Active.u16SubSource
            , ETG_CENUM(clStackRules::group_t, enGroup)));

      dp_tclAudioStackDPAudioStack_LastSource_Class oLastSource_Class;
      oLastSource_Class.vSetData(m_enSource_Entertainment_Active.enSourceClass);
      dp_tclAudioStackDPAudioStack_LastSource_SubID oLastSource_SubID;
      oLastSource_SubID.vSetData(m_enSource_Entertainment_Active.u16SubSource);
      // Do special group based tasks
      if (enGroup == clStackRules::groupRadio)
      {
         ETG_TRACE_USR4(("vAudioSourceOn(): RadioGroup "));
         //AudioComponent::enAudioComponent enDP_CurrentAudioComponent = AudioComponent::RADIO; // RADIO
      }
      else if (enGroup == clStackRules::groupMedia)
      {
         ETG_TRACE_USR4(("vAudioSourceOn(): MediaGroup "));
         //AudioComponent::enAudioComponent enDP_CurrentAudioComponent = AudioComponent::MEDIA; // MEDIA
      }
   }else{
      // Update the DP values, which will trigger the screen-changes and animations
      ETG_TRACE_USR3(("vAudioSourceOn(): Step 2: Ignored due to SourceClassType %d"
            , ETG_CENUM(clStackRules::group_t, enGroup)));
   }
   // ******
   // STEP 3
   // ******
   // Request Source Change on AudioSourceController
   if (enType == clStackRules::typeBg)
   {
      if(clAudioSourceController::getInstance().bSELECT(srcID, u32UserData))
      {
         return AUDIOSM_OK;
      }
   }
   else
   if (enType == clStackRules::typeMix)
   {
      if(clAudioSourceController::getInstance().bMIX(srcID, u32UserData))
      {
         return AUDIOSM_OK;
      }
   }
   else
   {
      if(clAudioSourceController::getInstance().bON(srcID, u32UserData))
      {
         return AUDIOSM_OK;
      }
   }
   return AUDIOSM_ERROR;
}


clAudioSMEngine::enErrorCode clAudioSMEngine::Source_AvailabilityChange(SourceID srcID,
                                                        clAudioSource::enSourceAvailability availability,
                                                        clAudioSource::enSourceAvailabilityReason reason)
{
   ETG_TRACE_USR4(("Source_Availability: SourceClass: %d subID %#x(%d), Availability: %d, Reason: %d"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
            , (tU16)srcID.u16SubSource
            , (tU16)srcID.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, availability)
            , ETG_CENUM(clAudioSource::enSourceAvailabilityReason, reason)
            ));
   //get an Instance
   clAudioSource* pAudSrc = clAudioSourceFactory::getAudioSource(srcID);
   //Source is not available at all
   if(pAudSrc == NULL)
   {
      return clAudioSMEngine::AUDIOSM_ERROR;
   }
   //Set the Availability according received
   pAudSrc->vSourceAvailablilityChange(availability, reason);
   return clAudioSMEngine::AUDIOSM_OK;
}


/*************************************************************************
* METHOD:         u8GetNextAvailableEntertainmentSource
*
* DESCRIPTION:    Iterate through the list of entertainment sources,
*                 defined in the specified sequence and find the next available one.
*
* PARAMETER:      u8CurrentAudioSource
*                 pau8ToggleSequence      defines the sequence descriptor to iterate through
*                 u8SizeToggleSequence    defines the size of the sequence descriptor
* RETURNVALUE:    next available ent-source or NONE if no ent-source
*                 available
*
************************************************************************/
SourceID clAudioSMEngine::GetNextAvailableEntertainmentSource(SourceID srcID)
{
   ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Find next Source after SourceClass: %d subID %#x(%d) "
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
            , (tU16)srcID.u16SubSource
            , (tU16)srcID.u16SubSource));
   SourceID nextSrcID ( AudioSources::NONE,0);
   clAudioSource* pSrc = clAudioSourceFactory::getAvailableAudioSource(srcID.enSourceClass);

   if(pSrc != NULL)
   {
      ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Next Source: %d subID %#x(%d) "
                  ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(pSrc->sGetId().enSourceClass))
                  , (tU16)pSrc->sGetId().u16SubSource
                  , (tU16)pSrc->sGetId().u16SubSource));
      return pSrc->sGetId();
   }else{
      //If we cannot find any available
      while(pSrc == NULL)
      {

         AudioSources::enAudioSources nextSrcClass =
                              clTogglingSequences::GetNextSourceClass(srcID.enSourceClass, AudioToggleGroups::RMT_CTRL);
         ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Search in next SourceClass Source: %d "
                           ,ETG_CENUM(AudioSources::enAudioSources,nextSrcClass)));
         pSrc = clAudioSourceFactory::getAvailableAudioSource(nextSrcClass);
      }
      ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Next Source: %d subID %#x(%d) "
                        ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(pSrc->sGetId().enSourceClass))
                        , (tU16)pSrc->sGetId().u16SubSource
                        , (tU16)pSrc->sGetId().u16SubSource));
      return pSrc->sGetId();
   }
}

/*************************************************************************
* METHOD:         vOnAudioServiceStateChanged
*
* DESCRIPTION:    ! Under Construction !
*
*
* PARAMETER:      u8ServiceState
*
* RETURNVALUE:    -
*
************************************************************************/

tVoid clAudioSMEngine::vOnAudioServiceStateChanged(tU16 u16Service, tU8 u8ServiceState)
{

   ETG_TRACE_USR4(("vOnAudioServiceStateChanged Service:%d State:%d",
         u16Service,
         u8ServiceState));
// TODO SERVICE AVAILABILITY
   //
//   if((u16Service == CCA_C_U16_SRV_AUDIO_ROUTE_MGR) && (u8ServiceState == AMT_C_U8_SVCSTATE_NOT_AVAILABLE))
//   {
//      ETG_TRACE_USR1(("Set m_u8AudioRouteMgr_SrvState : Not Available"));
//      m_u8AudioRouteMgr_SrvState = AMT_C_U8_SVCSTATE_NOT_AVAILABLE;
//      ETG_TRACE_USR1((" RESET AUDIO STACK!"));
//      vResetStack();
//   }
//
//   if((u16Service == CCA_C_U16_SRV_AUDIO_ROUTE_MGR) && (u8ServiceState == AMT_C_U8_SVCSTATE_AVAILABLE))
//   {
//      ETG_TRACE_USR1(("Set m_u8AudioRouteMgr_SrvState : Available"));
//      m_u8AudioRouteMgr_SrvState = AMT_C_U8_SVCSTATE_AVAILABLE;
//   }
}

/*************************************************************************
* METHOD:         vAudioSourceStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceStarted(tU8 u8AudioSource) {}

/*************************************************************************
* METHOD:         vAudioSourceStopped
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceStopped(tU8 u8AudioSource){}

/*************************************************************************
* METHOD:         vAudioSourceInitStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceInitStarted(tU8 u8AudioSource) {}

/*************************************************************************
* METHOD:         vAudioSourceStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8NewEntertainmentSource
*                 bBackground
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vEntertainmentSourceExchanged(tU8 u8NewEntertainmentSource, tBool bBackground) {}

/*************************************************************************
* METHOD:         vAudioSourceOff
*
* DESCRIPTION:    Stop an FG Source
* PARAMETER:      tU8 u8AudioSource    the FG source to stop
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceOff(SourceID srcID, tU32 u32UserData)
{
   clAudioSource* poAudioSource = NULL;
   clStackRules::type_t enType;

   ETG_TRACE_USR4(("vAudioSourceOff: SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
            , srcID.u16SubSource
            , srcID.u16SubSource));

   poAudioSource = clAudioSourceFactory::getAudioSource(srcID);
   enType = poAudioSource->getType();

   // Delegate Request to AudioSourceController
   if(enType == clStackRules::typeMix)
   {
      clAudioSourceController::getInstance().bMIXOFF(srcID, u32UserData);
   }
   else if(enType == clStackRules::typeFg)
   {
      clAudioSourceController::getInstance().bOFF(srcID, 0);
   }
   else
   {
      ETG_TRACE_ERR(("vAudioSourceOff: ONLY ALLOWED FOR Mix AND Fg SOURCES: conflict SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcID.enSourceClass))
         , srcID.u16SubSource
         , srcID.u16SubSource));
   }
}

/*************************************************************************
* METHOD:         vDiagnosisAudioSourceOn
*
* DESCRIPTION:    Activate an AudioSource for Diagnosis/RemoteControl
*                 AudioSource is started with force-option, i.e. a play
*                 request is sent to FC-Audio without any prio check and
*                 without any media-available check!
*                 The normal AudioStack is frozen.
* PARAMETER:      tU8 u8AudioSource    the Diagnosis Audio source to start
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisAudioSourceOn(tU8 u8AudioSource, tBool bForce)
{
   ETG_TRACE_ERR(("vDiagnosisAudioSourceOn: actually not supported: source %d(ID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(u8AudioSource))
            , u8AudioSource
            , u8AudioSource));
}

/*************************************************************************
* METHOD:         vDiagnosisAudioSourceOff
*
* DESCRIPTION:    Stop an AudioSource for Diagnosis/RemoteControl
*                 AudioSource is stopped with force-option, i.e. a stop
*                 request is sent to FC-Audio without any prio check!
* PARAMETER:      tU8 u8AudioSource    the Diagnosis Audio source to stop
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisAudioSourceOff(tU8 u8AudioSource, tBool bForce)
{
   ETG_TRACE_ERR(("vDiagnosisAudioSourceOff: actually not supported: source %d(ID %#x(%d)"
               ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(u8AudioSource))
               , u8AudioSource
               , u8AudioSource));
}

/*************************************************************************
* METHOD:         vDiagnosisOff
*
* DESCRIPTION:    Stop Diagnosis Mode
*                 If a Diagnosis AudioSource is currently active, it is
*                 stopped.
*                 The normal Audio Stack is recovered.
* PARAMETER:
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisOff()
{
   ETG_TRACE_ERR(("vDiagnosisOff: actually not supported"));
}

/*************************************************************************
* METHOD:         bIsSystemFree
*
* DESCRIPTION:    Asks CM-System, if PIN-Code Lock is active
* PARAMETER:
*
* RETURNVALUE:    tBool: True, if PIN-Code is unlocked, False otherwise
*
************************************************************************/
tBool clAudioSMEngine::bIsSystemFree()
{
   ETG_TRACE_USR1(("bIsSystemFree: TRUE"));
   return TRUE;
}

/*************************************************************************
* METHOD:         bIsEntertainmentSourceAvailable
*
* DESCRIPTION:    there are 2 types of Entertainment sources: Radio and Media
*                 Radio sources are available always.
*                 Media sources availability is given by bIsMediaSourceAvailable()
*
* PARAMETER:       u8AudioSource
*
* RETURNVALUE:     true, if the specified Entertainment source is available
*
*
************************************************************************/
tBool clAudioSMEngine::bIsSourceAvailable(SourceID audioSource)
{
   tBool retVal = FALSE;
   clAudioSource* pAudSrc = clAudioSourceFactory::getAudioSource(audioSource);
   if(pAudSrc != NULL)
   {
      if((pAudSrc->enIsSourceAvailable() == clAudioSource::available)
            ||(pAudSrc->enIsSourceAvailable() == clAudioSource::unkonwn))
      {
         retVal = TRUE;
      }
   }
   ETG_TRACE_USR2(("bIsSourceAvailable: SourceClass: %d subID %#x(%d), Availability: %d"
            , ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(audioSource.enSourceClass))
            , audioSource.u16SubSource
            , audioSource.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, pAudSrc->enIsSourceAvailable())));
   return retVal;
}

/*************************************************************************
* METHOD:         vSetRunlevel
*
* DESCRIPTION:    This is the entrypoint for CM-Startup, which controls
*                 the powerstates and runlevels of the AudioSystem via
*                 this method.
*                 --------------------------------------------------------
*                 Runlevel 0:
*                    - Audio is muted.
*                    - AudioStack is reseted
*                    - The virtual Source: System-Mute is on stack. This
*                    ensures the audio system to remain muted, even if
*                    source-switch requests are applied.
*                    - AudioSource-ON() is allowed. This is required to enable
*                    early Source-Switch requests. The requested sources
*                    are put to the stack but not yet started
*                 --------------------------------------------------------
*                 Runlevel 1:
*                    - This is the runlevel for limited Phone mode
*                    - All Audiosources except phone are muted
*                 --------------------------------------------------------
*                 Runlevel 2:
*                    - This is the runlevel for limited Entertainment audio
*                    - All Radio Sources are enabled
*                    - All Media Sources are still disabled
*                 --------------------------------------------------------
*                 Runlevel 3:
*                    - This is the runlevel for full audio
*                    - All Audio Sources are enabled
*
*
* PARAMETER:      tU8 u8NewRunlevel
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::SetRunlevel(tU8 u8NewRunlevel)
{
   tU8 u8OldRunlevel = m_u8Runlevel;
   m_u8Runlevel = u8NewRunlevel;

   //register static AudioSources at first
   if (u8OldRunlevel == 0 && u8NewRunlevel > 0)
   {
     clAudioSourceFactory::Init();
   }

   tU8 au8Data[2]={u8OldRunlevel, u8NewRunlevel};
   ETG_TRACE_USR2(("vSetRunlevel: Old Runlevel: %d New Runlevel: %d"
         ,u8OldRunlevel
         ,u8NewRunlevel));

   switch (u8NewRunlevel)
   {
   case 0:
      {
         if (u8OldRunlevel > 0)
         {
            // Reset Stack
            vResetStack();
            AudioSourceOn(SourceID(AudioSources::MUTE_SYSTEM,0));
            AudioSourceOn(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 1:
      {
         if (u8OldRunlevel < 1)
         {
            // enable limited phone audio
            vAudioSourceOff(SourceID(AudioSources::MUTE_PIN, 0));
            vAudioSourceOff(SourceID(AudioSources::MUTE_SYSTEM,0));
         }
         else if (u8OldRunlevel > 1)
         {
            // No Entertainment in runlevel 1. Only Phone.
            //vResetEntertainmentsource();
            AudioSourceOn(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 2:
      {
         if (u8OldRunlevel < 1)
         {
            vAudioSourceOff(SourceID(AudioSources::MUTE_SYSTEM,0));
         }
         if (u8OldRunlevel < 2)
         {
            // Restore Entertainmentsource (limited to radio) and persistent Pause-state
            vRestoreEntertainmentsource();
            vRestorePersistentMuteState();
            vAudioSourceOff(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 3:
      {
         if (u8OldRunlevel < 3)
         {
            time(&m_stInitTime);
            // Restore Entertainmentsource (radio and media)and persistent Pause-state
            vRestoreEntertainmentsource();
            vRestorePersistentMuteState();
            vAudioSourceOff(SourceID(AudioSources::MUTE_SYSTEM,0));
            vAudioSourceOff(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   default:
      // invalid runlevel
      break;
   }
}

tU8 clAudioSMEngine::GetRunlevel()
{

   return m_u8Runlevel;
}

tVoid clAudioSMEngine::ListSources()
{
   clAudioSourceFactory::ListSources();
}


tVoid clAudioSMEngine::vResetStack()
{

   clAudioSourceController::getInstance().vResetStack();
   // place additionally to a clean stack a Pin-Mute to ensure Pin-lock before
   // XML-Start, because XML locks the PIN-lock too late at startup.
   clAudioSourceController::getInstance().bON(SourceID(AudioSources::MUTE_PIN,0), 0);
}


/*************************************************************************
* METHOD:         vRestorePersistentMuteState
*
* DESCRIPTION:    Recovers the Persistent MuteState
*                 The Persistent MuteState is the user-requested Mute
*                 which pauses Entertainmentsources. (MFL, HK-Phone, ...)
* PARAMETER:      -
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vRestorePersistentMuteState()
{
   ETG_TRACE_USR1(("vRestorePersistentMuteState: ALWAYS NO MUTE STATE DUE TO MISSING PERSISTANCE MEMORY"));
}

time_t  clAudioSMEngine::GetTime_Init()
{
   return m_stInitTime;
}

/*************************************************************************
* METHOD:         vRestoreEntertainmentsource
*
* DESCRIPTION:    Recovers the LSM Entertainmentsource
*                 If runlevel is < 3, a Media-LSM will not be started
* PARAMETER:      -
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vRestoreEntertainmentsource()
{
   ETG_TRACE_USR4(("vRestoreEntertainmentsource:" ));
   time_t TheTimeIsNow;
   time(&TheTimeIsNow);
   tDouble SecSinceRunlvl3 = difftime(TheTimeIsNow,m_stInitTime);
   ETG_TRACE_USR4(("vRestoreEntertainmentsource: Seconds since Runlevel 3: %f",SecSinceRunlvl3 ));

   DP_vCreateDatapool();
   //Get SourceClass
   dp_tclAudioStackDPAudioStack_LastSource_Class   oLastSource_Class;
   tU8 lastSourceClass = 0;
   oLastSource_Class.u8GetData(lastSourceClass);
   //Get SubID
   dp_tclAudioStackDPAudioStack_LastSource_SubID   oLastSource_SubID;
   tU16 u16SubID = 0;
   oLastSource_SubID.s32GetData(u16SubID);

   ETG_TRACE_USR4(("vRestoreEntertainmentsource: Read from DP SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(lastSourceClass))
         , u16SubID
         , u16SubID));

   SourceID lastSource (static_cast<AudioSources::enAudioSources>(lastSourceClass),u16SubID);
   m_enSource_Entertainment_LSM = lastSource;
   //Check if last source is already there
   if(clAudioSourceFactory::isSourceInstantiated(lastSource) == NULL)
   {
      //source is not there yet, we create it with unkown
      //availability
      //source will itself check if timeout occurs or already tiemd out
      //then, if it is in unkown state, it will mark itself as unavailable
      clAudioSource* pAudSrc = clAudioSourceFactory::getAudioSource(lastSource);
      if(pAudSrc != NULL)
      {
         pAudSrc->vSourceAvailablilityChange(clAudioSource::unkonwn, clAudioSource::samemedia);
      }
   }

   if(lastSourceClass == 0)
   {
      ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restore Default source due to default value of datapool" ));
      lastSource.enSourceClass = AudioSources::TUNER_FM;
      lastSource.u16SubSource = 0;
   }

   if (!bIsSourceAvailable(lastSource))
   {
      ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restore Default source due to availability" ));
      lastSource.enSourceClass = AudioSources::TUNER_FM;
      lastSource.u16SubSource = 0;
   }

   ETG_TRACE_USR1(("vRestoreEntertainmentsource: Adding MainConnection SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(lastSource.enSourceClass))
         , lastSource.u16SubSource
         , lastSource.u16SubSource));
   clAudioSource* pGSrc =  clAudioSourceFactory::getAudioSource(lastSource);
   if(pGSrc == NULL )
   {
      ETG_TRACE_ERR(("vRestoreEntertainmentsource: No AudioSource Object found"));
      return;
   }
   AudioSourceOn(SourceID(static_cast<AudioSources::enAudioSources>(lastSourceClass),u16SubID), 0);
}

}//namespace



